/*
	Socket.cpp
	This file serves as the base for all sockets.
	Copyright 2002 Ben Everett and Chris Horlick.
	All Rights Reserved.
*/

#include "pch.h"

// Engine Includes
#include "Socket.h"

using namespace CPPMUD;

// Constructor & Deconstructor
CClientDescriptor::CClientDescriptor()
{
}

CClientDescriptor::~CClientDescriptor()
{
}

CServerSocket::CServerSocket()
{
	m_nMaxClients = 1;
	m_nNextClientID = 0;

	m_vecClients.clear();
}

CServerSocket::~CServerSocket()
{
}

// Functions
const int CClientDescriptor::GetLineFromInputBuffer(char *czOutput, int nMaxLength)
/*
	Obtains an entire line from the incoming data buffer.
	PRE: Server has been initialized properly, czOutput is a valid pointer to a character array, 
		nMaxLength is the amount of character's czOutput can hold.
	POST: Line copied from incoming data buffer to czOutput and the length of the line returned.
*/
{
	char *cLineEnd = strstr(m_czIncomingData, "\n");
	int nLen = strlen(m_czIncomingData);
	int nLineEnd = cLineEnd - m_czIncomingData + 1;

	// Make sure we found an line in the input buffer.
	if (cLineEnd == NULL)
		return 0;

	// Copy the line from the input buffer to czOutput.
	for (int i = 0; i < nLineEnd; ++i)
	{
		// Make sure we haven't surpassed the length of czOutput.
		if (i >= nMaxLength)
			break;

		czOutput[i] = m_czIncomingData[i];
	}

	// Remove the line from the incoming data buffer.
	// *NOTE* In the MUD we truncate all data that surpasses czOutput, you may want to change this 
	//    for your purposes.
	memmove(m_czIncomingData, m_czIncomingData + nLineEnd, 2048 - nLineEnd);
	memset(m_czIncomingData + (nLen - nLineEnd), 0, sizeof(char) * (nLen - nLineEnd));

	return i;
}

ErrRet CServerSocket::CheckForNewData()
/*
	Checks for pending client connection attempts and incoming/outgoing client data.
	PRE: Server has been initialized properly.
	POST: New clients connected and incoming/outgoing data flags have been set for each client.
*/
{
	ErrRet retVal;
	fd_set fdExcept, fdRead, fdWrite;
	int i = 0;
	SOCKET sMaxDesc;
	timeval timeNone;

	timeNone.tv_sec = timeNone.tv_usec = 0;

	FD_ZERO(&fdExcept);
	FD_ZERO(&fdRead);
	FD_ZERO(&fdWrite);

	FD_SET(m_sSocket, &fdExcept);
	FD_SET(m_sSocket, &fdRead);
	FD_SET(m_sSocket, &fdWrite);

	sMaxDesc = m_sSocket;

	for (i = 0; i < m_vecClients.size(); ++i)
	{
		FD_SET(m_vecClients[i].m_sSocket, &fdRead);
		FD_SET(m_vecClients[i].m_sSocket, &fdWrite);

		// Clear the incoming/outgoing data flags.
		m_vecClients[i].m_bInputPending = false;

		// Make sure we have the highest socket descriptor.
		if (m_vecClients[i].m_sSocket > sMaxDesc)
			sMaxDesc = m_vecClients[i].m_sSocket;
	}

	select(sMaxDesc + 1, &fdRead, &fdWrite, &fdExcept, &timeNone);

	// Check to make sure the server socket hasn't had an exception.
	if (FD_ISSET(m_sSocket, &fdExcept))
	{
		std::cout << "The server socket has encountered an exception, aborting..." << std::endl;

		return errSocketException;
	}

	// Check for incoming connections.
	if (FD_ISSET(m_sSocket, &fdRead))
	{
		retVal = NewClient();
		if (retVal != errNone)
			return retVal;
	}

	// Check all client sockets for incoming/outgoing data.
	for (i = 0; i < m_vecClients.size(); ++i)
	{
		// Incoming data, set incoming data flag on client.
		if (FD_ISSET(m_vecClients[i].m_sSocket, &fdRead))
		{
			m_vecClients[i].m_bInputPending = true;

			retVal = ReadData(m_vecClients[i].m_czIncomingData, 2048, m_vecClients[i].m_nClientID);
			if ((retVal != errNone) && (retVal == errSocketTimeOut))
			{
				CloseClient(m_vecClients[i].m_nClientID);

				continue;
			}
		}

		// Send as much outgoing data as possible to the client.
		retVal = SendData(m_vecClients[i].m_nClientID);
		if ((retVal != errNone) && (retVal == errSocketTimeOut))
			{
				CloseClient(m_vecClients[i].m_nClientID);

				continue;
			}

		// Close the connection if the client is overflow and all data has been sent.
		if (m_vecClients[i].m_csState == csOverflow)
		{
			if (strlen(m_vecClients[i].m_czOutgoingData) == 0)
			{
				CloseClient(m_vecClients[i].m_nClientID);

				continue;
			}
		}
	}

	return errNone;
}

ErrRet CServerSocket::NewClient()
/*
	Handles a client connection to the server.
	PRE: Server has been initialized properly.
	POST: Client connected to the server.
*/
{
	CClientDescriptor cdClient;
	char *czServerFull = "We are sorry for the inconvenience, but the server is currently full.\nPlease try again later.\n";
	int nLen = sizeof(cdClient.m_saClientInf);

	// Set the connection state and client ID.
	cdClient.m_nClientID = m_nNextClientID++;

	memset(cdClient.m_czIncomingData, 0, sizeof(char) * 2048);
	memset(cdClient.m_czOutgoingData, 0, sizeof(char) * 2048);

	// Accept the incoming connection.
	// *NOTE* May want to make this a virtual function and let CBSDSocket and CWinSocket handle this.
#ifdef _WIN32
	cdClient.m_sSocket = accept(m_sSocket, (sockaddr*)&cdClient.m_saClientInf, &nLen);
#else
	cdClient.m_sSocket = accept(m_sSocket, (sockaddr*)&cdClient.m_saClientInf, (socklen_t*)&nLen);
#endif

	// Server is full, close the connection.
	if (m_vecClients.size() >= m_nMaxClients)
	{
		std::cout << "Server full, refusing access to incoming client #" << cdClient.m_nClientID << 
			"." << std::endl;

		cdClient.m_csState = csOverflow;

		strcpy(cdClient.m_czOutgoingData, czServerFull);
	}
	else
	{
		// Set the connection state to connected.
		cdClient.m_csState = csConnected;

		std::cout << "Access to incoming connection permitted, new client #" << cdClient.m_nClientID << 
			"." << std::endl;

		strcpy(cdClient.m_czOutgoingData, "Connected.\r\n");
	}

	// Set the client socket to non-blocking.
	NonBlocking(cdClient.m_sSocket);

	// Append the client to the client list.
	m_vecClients.push_back(cdClient);

	return errNone;
}

ErrRet CServerSocket::SendData(char *czData, int nBufferSize, int nClientID)
/*
	Appends data to a client's output buffer.
	PRE: czData is a valid pointer to a character array, nBufferSize is the length of czData, 
		nClientID is whom the data is intended for.
	POST: New clients connected and incoming/outgoing data flags have been set for each client.
*/
{
	int nIndex = GetIndexFromClientID(nClientID);
	int nOutputLen = strlen(m_vecClients[nIndex].m_czOutgoingData);

	// Make sure there is room for the outgoing data.
	if ((nBufferSize + nOutputLen) > 2048)
		return errClientFullBuffer;

	sprintf(m_vecClients[nIndex].m_czOutgoingData + nOutputLen, "%s", czData);

	return errNone;
}

int CServerSocket::GetIndexFromClientID(int nClientID)
/*
	Returns an index into the client vector.
	PRE: None.
	POST: A index is returned from the client ID.
*/
{
	// Find the index from the client ID.
	for (int i = 0; i < m_vecClients.size(); ++i)
	{
		if (m_vecClients[i].m_nClientID == nClientID)
			return i;
	}

	return -1;
}